import os
import shutil
from datetime import datetime

# --- 設定項目 ---
# PDFファイルが格納されている「作業用」フォルダのパスを指定してください
# スクリプトと同じ階層にある場合は '作業用' のままで大丈夫です
WORK_FOLDER_PATH = '作業用'
# ----------------

def organize_pdf_files():
    """
    「作業用」フォルダ内のPDFファイルを、ファイル名に基づいて
    「本日日付/名称」のフォルダ構造に整理して移動します。
    """
    # 1. 本日日付のフォルダ名を作成 (例: 2025年10月07日)
    today_str = datetime.now().strftime('%Y年%m月%d日')
    date_folder_path = os.path.join(os.getcwd(), today_str)

    # 本日日付のフォルダがなければ作成
    os.makedirs(date_folder_path, exist_ok=True)
    print(f"作成/確認したフォルダ: {date_folder_path}")

    # 2. 「作業用」フォルダが存在するかチェック
    if not os.path.isdir(WORK_FOLDER_PATH):
        print(f"エラー: '{WORK_FOLDER_PATH}' フォルダが見つかりません。")
        return

    # 3. 「作業用」フォルダ内のファイルを処理
    for filename in os.listdir(WORK_FOLDER_PATH):
        # PDFファイルのみを対象とする
        if filename.endswith('.pdf'):
            try:
                # 4. ファイル名から「名称」を抽出
                # 最初の"_"で分割し、2番目の要素以降を結合して名称とする
                parts = filename.split('_', 1)
                if len(parts) > 1:
                    # ".pdf"拡張子を除去して名称を取得
                    name = parts[1].replace('.pdf', '')

                    # 5. 名称ごとの子フォルダを作成
                    name_folder_path = os.path.join(date_folder_path, name)
                    os.makedirs(name_folder_path, exist_ok=True)

                    # 6. PDFファイルを移動
                    source_path = os.path.join(WORK_FOLDER_PATH, filename)
                    destination_path = os.path.join(name_folder_path, filename)

                    shutil.move(source_path, destination_path)
                    print(f"移動しました: {filename} -> {name_folder_path}")
                else:
                    print(f"スキップしました（名前に'_'が含まれていません）: {filename}")
            except Exception as e:
                print(f"エラーが発生しました: {filename} の処理中に問題が発生しました。 - {e}")

    print("\nすべての処理が完了しました。")

if __name__ == '__main__':
    organize_pdf_files()