import fitz  # PyMuPDFライブラリ
import csv   # CSVファイルを扱うためのライブラリ
import os    # ファイルパスを操作するためのライブラリ

# ◆ 設定する箇所 ◆
# ----------------------------------------------------
# 1. テンプレートとなるPDFファイルの名前
input_pdf_path = 'your_original_file.pdf'  # ★ご自身のPDFファイル名に変更してください

# 2. 読み込むCSVファイルの名前
input_csv_path = 'data.csv'

# 3. 出力するPDFを保存するフォルダ名（なければ自動で作成します）
output_folder = 'output_pdfs'
# ----------------------------------------------------


# --- ここから下はプログラムの処理です ---

def create_pdfs_from_csv():
    # 出力用フォルダがなければ作成
    if not os.path.exists(output_folder):
        os.makedirs(output_folder)
        print(f"フォルダ '{output_folder}' を作成しました。")

    try:
        # ★★★ CSVファイルを開く文字コードを 'cp932' に変更 ★★★
        with open(input_csv_path, mode='r', encoding='cp932') as csvfile:
            # ヘッダーをキーとする辞書としてCSVを読み込む
            reader = csv.DictReader(csvfile)

            # CSVの各行（データ1件分）に対してループ処理
            for index, row_data in enumerate(reader):
                # テンプレートPDFを開く
                doc = fitz.open(input_pdf_path)
                page = doc[0] # 最初のページ

                # PDFのフォームフィールドをループ
                for field in page.widgets():
                    # フィールド名がCSVのヘッダー（row_dataのキー）に存在するかチェック
                    if field.field_name in row_data:
                        # 対応するデータをフィールドに設定
                        field.field_value = row_data[field.field_name]
                        # 見た目を更新
                        field.update()
                
                # 保存するファイル名を生成 (例: 1_山田 太郎.pdf)
                # ファイル名に使えない文字はアンダースコアに置換
                safe_name = row_data.get('名前漢字', f'data_{index + 1}').replace('/', '_').replace('\\', '_')
                output_pdf_name = f"{index + 1}_{safe_name}.pdf"
                output_pdf_path = os.path.join(output_folder, output_pdf_name)

                # 変更を保存
                doc.save(output_pdf_path)
                doc.close()
                
                print(f"[{index + 1}件目] '{output_pdf_path}' を作成しました。")
            
            print("\nすべての処理が完了しました。")

    except FileNotFoundError:
        print(f"エラー: '{input_pdf_path}' または '{input_csv_path}' が見つかりません。")
        print("ファイルがプログラムと同じ階層にあるか確認してください。")
    except UnicodeDecodeError:
        print("エラー: CSVファイルの文字コードが正しくありません。")
        print("ファイルをExcelで開き、'CSV UTF-8 (コンマ区切り)' で保存し直すか、プログラムのエンコーディング指定を確認してください。")
    except Exception as e:
        print(f"エラーが発生しました: {e}")

# プログラムを実行
create_pdfs_from_csv()