import pandas as pd
from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.common.exceptions import WebDriverException, NoSuchElementException
import time
import sys

# --- 設定項目 ---

# 顧客データが記載されているExcelファイルのパス
EXCEL_FILE_PATH = 'customer.xlsx'
# Excelシート名
EXCEL_SHEET_NAME = 'customer'

# PHPプログラムが配置されているURLを指定してください。
# ★★★ ご指定のURLに合わせて修正しました ★★★
BASE_URL = 'http://localhost/'

# ----------------

def main():
    """
    Excelファイルから顧客データを読み込み、Webフォームに自動登録するメイン処理。
    """
    # 1. Excelファイルから顧客データを読み込む
    try:
        # 1行目はヘッダー行とみなし、2行目からデータを読み込む
        df = pd.read_excel(
            EXCEL_FILE_PATH,
            sheet_name=EXCEL_SHEET_NAME,
            header=0,
            # 列のデータ型を文字列として読み込む（電話番号の先頭ゼロ落ち対策）
            dtype={
                '顧客名': str,
                'フリガナ': str,
                '電話番号': str,
                'メールアドレス': str,
                '住所': str,
            }
        )
        # Excelの列名をプログラムで使うキーに変換
        df.columns = ['name', 'name_kana', 'phone', 'email', 'address']

    except FileNotFoundError:
        print(f"エラー: Excelファイル '{EXCEL_FILE_PATH}' が見つかりません。")
        sys.exit(1)
    except Exception as e:
        print(f"Excelファイルの読み込み中にエラーが発生しました: {e}")
        sys.exit(1)

    # 2. Selenium WebDriverを初期化
    try:
        driver = webdriver.Chrome()
    except WebDriverException as e:
        print("WebDriverの起動に失敗しました。")
        print("最新のSeleniumライブラリがインストールされているか確認してください。")
        print("それでも失敗する場合、手動でChromeDriverをダウンロード・配置する必要があるかもしれません。")
        print(f"エラー詳細: {e}")
        sys.exit(1)

    print("ブラウザを起動し、データ登録処理を開始します。")

    # 3. 顧客データを行ごとに処理
    for index, row in df.iterrows():
        # 'name'（顧客名）が空(NaN)または空文字列の場合はスキップ
        if pd.isna(row['name']) or not str(row['name']).strip():
            print(f"Excelの {index + 2} 行目: 顧客名が空のためスキップします。")
            continue

        try:
            # 新規登録ページにアクセス
            create_page_url = BASE_URL + 'create.php'
            driver.get(create_page_url)
            time.sleep(1) # ページの読み込みを待機

            customer_name = str(row.get('name', ''))
            print(f"Excelの {index + 2} 行目のデータを登録中: {customer_name}")

            # Webフォームにデータを入力（値が空(NaN)の場合は空文字に変換）
            driver.find_element(By.ID, 'name').send_keys(customer_name)
            driver.find_element(By.ID, 'name_kana').send_keys(str(row.get('name_kana', '')) if pd.notna(row.get('name_kana')) else '')
            driver.find_element(By.ID, 'address').send_keys(str(row.get('address', '')) if pd.notna(row.get('address')) else '')
            driver.find_element(By.ID, 'phone').send_keys(str(row.get('phone', '')) if pd.notna(row.get('phone')) else '')
            driver.find_element(By.ID, 'email').send_keys(str(row.get('email', '')) if pd.notna(row.get('email')) else '')

            # 登録ボタンをクリック
            submit_button = driver.find_element(By.XPATH, '//button[@type="submit"]')
            submit_button.click()

            # サーバーの処理と次のページの読み込みを待機
            time.sleep(1)

        except NoSuchElementException as e:
            print(f"エラー: ページ上の要素（フォーム項目やボタン）が見つかりませんでした。")
            print(f"URLが正しいか、ページの構成が変更されていないか確認してください。({create_page_url})")
            break # エラーが発生した場合は処理を中断
        except Exception as e:
            print(f"Excelの {index + 2} 行目 ({customer_name}) の処理中に予期せぬエラーが発生しました: {e}")
            continue # エラーが発生しても次の行の処理を試みる

    # 4. ブラウザを閉じる
    print("すべてのデータの処理が完了しました。")
    driver.quit()

if __name__ == '__main__':
    main()