import os
import sys
import pytesseract
from pdf2image import convert_from_path
import pikepdf

# ----------------------------------------------------------------
# 事前設定: ご自身の環境に合わせて以下の2つのパスを編集してください
# ----------------------------------------------------------------

# 1. Tesseract-OCRの実行ファイルのパス (ステップ1で確認したパス)
#    バックスラッシュ(\)は2つ重ねるか、先頭に r を付けてください。
#    例: r'C:\Program Files\Tesseract-OCR\tesseract.exe'
TESSERACT_CMD_PATH = r'C:\Program Files\Tesseract-OCR\tesseract.exe'

# 2. Popplerのbinフォルダのパス (ステップ1で確認したパス)
#    例: r'C:\Users\YourUser\Downloads\poppler-24.02.0-win-64\Library\bin'
POPPLER_PATH = r'C:\Program Files\poppler-25.07.0\Library\bin'  # ← ここにPopplerのbinフォルダのパスを入力してください

# ----------------------------------------------------------------


def check_dependencies():
    """実行に必要なプログラムのパスとデータが正しく設定されているか確認する関数"""
    is_ok = True
    if not os.path.exists(TESSERACT_CMD_PATH):
        print(f"エラー: Tesseract-OCRが見つかりません。")
        print(f"指定されたパス: {TESSERACT_CMD_PATH}")
        print("プログラム内の TESSERACT_CMD_PATH を正しいパスに設定してください。")
        is_ok = False

    if not POPPLER_PATH or not os.path.isdir(POPPLER_PATH):
        print(f"エラー: Popplerのbinフォルダのパスが設定されていません。")
        print(f"指定されたパス: {POPPLER_PATH}")
        print("プログラム内の POPPLER_PATH にbinフォルダのパスを設定してください。")
        is_ok = False

    tesseract_dir = os.path.dirname(TESSERACT_CMD_PATH)
    tessdata_dir = os.path.join(tesseract_dir, 'tessdata')

    if not os.path.isdir(tessdata_dir):
         print(f"エラー: Tesseractの'tessdata'フォルダが見つかりません。")
         print(f"探した場所: {tessdata_dir}")
         is_ok = False
    else:
        osd_data_path = os.path.join(tessdata_dir, 'osd.traineddata')
        if not os.path.exists(osd_data_path):
            print(f"エラー: 向き検出用の学習データ 'osd.traineddata' が見つかりません。")
            print(f"探した場所: {osd_data_path}")
            print("手順を参考に 'osd.traineddata' をダウンロードして配置してください。")
            is_ok = False
            
    return is_ok


def main():
    """メイン処理を実行する関数"""
    if not check_dependencies():
        input("\n設定を確認してプログラムを再起動してください。何かキーを押すと終了します...")
        sys.exit(1)

    pytesseract.pytesseract.tesseract_cmd = TESSERACT_CMD_PATH

    base_dir = os.getcwd()
    fax_dir = os.path.join(base_dir, 'FAX')
    work_dir = os.path.join(base_dir, '作業用')

    if not os.path.exists(work_dir):
        os.makedirs(work_dir)
        print(f"フォルダを作成しました: {work_dir}")

    if not os.path.exists(fax_dir):
        print(f"エラー: 「FAX」フォルダが見つかりません。")
        print("プログラムと同じ場所に「FAX」フォルダを作成してください。")
        input("\n何かキーを押すと終了します...")
        return

    try:
        pdf_files = [f for f in os.listdir(fax_dir) if f.lower().endswith('.pdf')]
    except FileNotFoundError:
        print(f"エラー: 「FAX」フォルダが見つかりません。")
        return

    if not pdf_files:
        print("「FAX」フォルダに処理対象のPDFファイルがありません。")
        input("\n何かキーを押すと終了します...")
        return

    pdf_files.sort(key=lambda f: os.path.getctime(os.path.join(fax_dir, f)))
    print(f"処理対象のファイル: {len(pdf_files)}件")
    print("-" * 40)

    for filename in pdf_files:
        input_path = os.path.join(fax_dir, filename)
        output_path = os.path.join(work_dir, filename)

        print(f"処理中: {filename}")

        try:
            with pikepdf.open(input_path, allow_overwriting_input=True) as original_pdf:
                with pikepdf.Pdf.new() as new_pdf:
                    images = convert_from_path(input_path, poppler_path=POPPLER_PATH)

                    for i, page_image in enumerate(images):
                        page_num = i + 1
                        print(f"  - {page_num}ページ目の向きを判定中...")
                        try:
                            # TesseractのOSD機能で文字の向きを検出
                            osd = pytesseract.image_to_osd(page_image, lang='osd', output_type=pytesseract.Output.DICT)
                            rotation = osd.get('rotate', 0)
                            
                            page = original_pdf.pages[i]
                            
                            if rotation != 0:
                                print(f"    -> {rotation}度回転します。")
                                page.rotate(rotation, relative=True)
                            else:
                                print(f"    -> 向きは正常です。")
                            
                            new_pdf.pages.append(page)

                        except pytesseract.TesseractError as e:
                            print(f"  エラー: {page_num}ページ目の向きを判定できませんでした。ページをそのままコピーします。")
                            print(f"  [Tesseract エラー詳細]: {e}")
                            new_pdf.pages.append(original_pdf.pages[i])

                    new_pdf.save(output_path)
                    print(f"保存しました: {output_path}")

        except Exception as e:
            print(f"エラー: {filename} の処理中に予期せぬ問題が発生しました。 詳細: {e}")
        
        print("-" * 40)

    print("すべての処理が完了しました。")
    input("\n何かキーを押すと終了します...")


if __name__ == '__main__':
    main()