import os
import subprocess
import sys
from pathlib import Path

# --- 設定項目 ---

# 1. 作業用フォルダの名前
WORK_DIR = Path("作業用")

# 2. FAX発信元が記載されたテキストファイルの名前
FAX_LIST_FILE = Path("FAX発信元リスト.txt")

# 3. 使用するGemini CLIのコマンド名
# .ps1 スクリプトのフルパスを指定
GEMINI_COMMAND = r"C:\Users\minak\AppData\Roaming\npm\gemini.ps1"

# --- プログラム本体 ---

def get_fax_sender_from_gemini(pdf_path, fax_list_content):
    """
    Gemini CLIをYOLOモードで呼び出してPDFからFAX送信元を取得します。

    Args:
        pdf_path (Path): 処理対象のPDFファイルパス。
        fax_list_content (str): FAX発信元リストのテキスト内容。

    Returns:
        str or None: 抽出した送信元名、またはエラーの場合はNone。
    """
    # Geminiに与える指示（プロンプト）
    prompt = f"""
あなたは高性能なAIアシスタントです。
添付されたPDFファイルはFAXの受信データです。内容を正確に読み取ってください。

そして、このFAXの送信元企業名を特定し、後述する「送信元候補リスト」の中から最も合致するものを一つだけ選び、その企業名のみを回答してください。
余計な説明や前置きは一切含めず、企業名だけを出力してください。

企業名が不明であったり、自信の無い場合は、「不明」と出力して下さい。
--- 送信元候補リスト ---
{fax_list_content}
--------------------
"""

    # Gemini CLIに渡すコマンドの引数リストを作成
    # .ps1スクリプトなので、powershell.exe経由で呼び出す
    command = [
        "powershell",
        "-File",
        GEMINI_COMMAND,
        "--yolo",
        prompt,
        str(pdf_path)
    ]

    try:
        print(f"  > Geminiに問い合わせ中 (YOLOモード): {pdf_path.name}")
        # Gemini CLIのコマンドを実行
        result = subprocess.run(
            command,
            capture_output=True,
            text=True,
            check=True,
            encoding='utf-8'
        )
        # Geminiからの標準出力を取得し、前後の空白や改行を削除
        sender_name = result.stdout.strip()
        # 複数行の応答が返ってきた場合に備えて、最初の1行のみ採用
        return sender_name.splitlines()[0]

    except FileNotFoundError:
        print(f"\nエラー: コマンド 'powershell' が見つかりません。", file=sys.stderr)
        print("PowerShellが正しくインストールされ、実行パスが通っているか確認してください。", file=sys.stderr)
        return None
    except subprocess.CalledProcessError as e:
        print(f"  > エラー: Gemini CLIの実行中にエラーが発生しました。", file=sys.stderr)
        print(f"  > 詳細: {e.stderr}", file=sys.stderr)
        return None
    except Exception as e:
        print(f"  > エラー: 予期せぬエラーが発生しました - {e}", file=sys.stderr)
        return None

def main():
    """
    メインの処理を実行します。
    """
    print("--- FAX受信PDFリネーム処理を開始します ---")

    # 前提条件のチェック
    if not WORK_DIR.is_dir():
        print(f"エラー: '{WORK_DIR}' フォルダが見つかりません。処理を中断します。")
        return
    if not FAX_LIST_FILE.is_file():
        print(f"エラー: '{FAX_LIST_FILE}' が見つかりません。処理を中断します。")
        return

    # FAX発信元リストを読み込む
    try:
        with open(FAX_LIST_FILE, "r", encoding='utf-8') as f:
            fax_list_content = f.read()
            fax_senders = [line.strip() for line in fax_list_content.strip().splitlines()]
        print(f"参照ファイル '{FAX_LIST_FILE}' を読み込みました。")
    except Exception as e:
        print(f"エラー: '{FAX_LIST_FILE}' の読み込みに失敗しました: {e}")
        return

    # 「作業用」フォルダ内のPDFファイルを作成日時の昇順で取得
    try:
        pdf_files = sorted(
            [p for p in WORK_DIR.glob("*.pdf")],
            key=lambda p: p.stat().st_ctime
        )
    except Exception as e:
        print(f"エラー: PDFファイルの取得中にエラーが発生しました: {e}")
        return

    if not pdf_files:
        print(f"'{WORK_DIR}' フォルダに処理対象のPDFファイルが見つかりませんでした。")
        return

    print(f"{len(pdf_files)}個のPDFファイルを処理します。\n")

    # 各PDFファイルを順に処理
    success_count = 0
    fail_count = 0
    for i, pdf_path in enumerate(pdf_files, 1):
        print(f"[{i}/{len(pdf_files)}] 処理中ファイル: {pdf_path.name}")

        # Gemini CLIでFAX送信元を取得
        sender_name = get_fax_sender_from_gemini(pdf_path, fax_list_content)

        if sender_name and sender_name in fax_senders:
            # 新しいファイル名を生成
            new_stem = f"{pdf_path.stem}_{sender_name}"
            new_path = pdf_path.with_stem(new_stem)

            # ファイル名変更
            try:
                pdf_path.rename(new_path)
                print(f"  > 成功: ファイル名を '{new_path.name}' に変更しました。\n")
                success_count += 1
            except Exception as e:
                print(f"  > エラー: ファイル名の変更に失敗しました - {e}\n", file=sys.stderr)
                fail_count += 1
        else:
            if not sender_name:
                print(f"  > 失敗: Geminiから送信元名を取得できませんでした。\n")
            else:
                print(f"  > 失敗: 取得した送信元 '{sender_name}' がリストに存在しません。\n")
            fail_count += 1

    print("--- 全ての処理が完了しました ---")
    print(f"成功: {success_count}件, 失敗: {fail_count}件")


if __name__ == "__main__":
    main()