import openpyxl
from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.chrome.service import Service as ChromeService
from webdriver_manager.chrome import ChromeDriverManager
import time
import os

# --- 設定項目 ---

# 顧客登録ページのURLを、あなたの環境に合わせて設定してください。
# 例: http://localhost/customer_app/create.php
CREATE_PAGE_URL = 'http://localhost/create.php'

# 読み込むExcelファイル名
EXCEL_FILE_NAME = 'customer.xlsx'

# 読み込むシート名
SHEET_NAME = 'customer'

# --- メイン処理 ---

def main():
    """
    Excelファイルから顧客データを読み込み、Webフォームに自動登録するメイン関数
    """
    # --- 1. Excelファイルの存在確認 ---
    if not os.path.exists(EXCEL_FILE_NAME):
        print(f"エラー: Excelファイル '{EXCEL_FILE_NAME}' が見つかりません。")
        print("このPythonスクリプトと同じフォルダにファイルを配置してください。")
        return

    # --- 2. WebDriverのセットアップ ---
    print("WebDriverをセットアップしています...")
    try:
        # ChromeDriverを自動的にダウンロードして設定します
        service = ChromeService(ChromeDriverManager().install())
        driver = webdriver.Chrome(service=service)
        driver.implicitly_wait(5) # 要素が見つかるまで最大5秒待機
    except Exception as e:
        print(f"WebDriverのセットアップ中にエラーが発生しました: {e}")
        print("Google Chromeがインストールされていることを確認してください。")
        return

    # --- 3. Excelファイルの読み込み ---
    print(f"'{EXCEL_FILE_NAME}' からデータを読み込んでいます...")
    try:
        workbook = openpyxl.load_workbook(EXCEL_FILE_NAME)
        sheet = workbook[SHEET_NAME]
    except KeyError:
        print(f"エラー: Excelファイルに '{SHEET_NAME}' という名前のシートが見つかりません。")
        driver.quit()
        return
    except Exception as e:
        print(f"Excelファイルの読み込み中にエラーが発生しました: {e}")
        driver.quit()
        return

    # --- 4. データ登録ループ処理 ---
    print("データの登録処理を開始します。")
    # 2行目から最終行までループ (ヘッダー行をスキップ)
    for i, row in enumerate(sheet.iter_rows(min_row=2, values_only=True), start=2):
        # Excelの列に合わせてデータを取得 (A列->0, B列->1, ...)
        # セルの値がNoneの場合は、空文字''に変換します
        customer_name = row[0] if row[0] is not None else ''
        name_kana = row[1] if row[1] is not None else ''
        phone_number = str(row[2]) if row[2] is not None else '' # 電話番号は文字列として扱う
        email_address = row[3] if row[3] is not None else ''
        address = row[4] if row[4] is not None else ''

        # 必須項目である顧客名が空の場合は、その行をスキップします
        if not customer_name:
            print(f"[{i}行目]: 顧客名が空のためスキップします。")
            continue

        try:
            # 登録ページにアクセス
            driver.get(CREATE_PAGE_URL)

            # フォームの各要素にデータを入力します
            # create.php の id 属性を指定しています
            driver.find_element(By.ID, 'name').send_keys(customer_name)
            driver.find_element(By.ID, 'name_kana').send_keys(name_kana)
            driver.find_element(By.ID, 'address').send_keys(address)
            driver.find_element(By.ID, 'phone').send_keys(phone_number)
            driver.find_element(By.ID, 'email').send_keys(email_address)

            # 登録ボタンをクリック (Bootstrapのクラス名などを参考に要素を特定)
            submit_button = driver.find_element(By.CSS_SELECTOR, 'button.btn-primary[type="submit"]')
            submit_button.click()

            print(f"[{i}行目]: '{customer_name}' のデータを登録しました。")

            # サーバーの処理と次のページ読み込みのために少し待機
            time.sleep(1)

        except Exception as e:
            print(f"[{i}行目]: '{customer_name}' の登録中にエラーが発生しました: {e}")
            # エラーが発生しても処理を継続します。中断したい場合は下の行のコメントを解除してください。
            # break

    # --- 5. 終了処理 ---
    print("すべてのデータの登録処理が完了しました。")
    driver.quit()

if __name__ == '__main__':
    main()